//FileDocumentWindow class.  Implements "nice" behavior for saving and quitting.

//******************************************************************************************************
//**** SYSTEM HEADER FILES
//******************************************************************************************************
#include <stdio.h>
#include <string.h>
#include <FilePanel.h>
#include <Path.h>
#include <Alert.h>


//******************************************************************************************************
//**** PROJECT HEADER FILES
//******************************************************************************************************
#include "FileDocumentWindow.h"
#include "DocumentApp.h"


//******************************************************************************************************
//**** FileDocumentWindow CLASS
//******************************************************************************************************
FileDocumentWindow::FileDocumentWindow(BRect a_frame, const char *a_filename, window_type a_type, uint32 a_flags, 
	uint32 a_workspace)
: BWindow(a_frame, GetLeafName(a_filename), a_type, a_flags, a_workspace)
{
	FDWConstructorWork(a_filename);
}


FileDocumentWindow::FileDocumentWindow(BRect a_frame, const char *a_filename, window_look a_look, window_feel a_feel, 
	uint32 a_flags, uint32 a_workspace)
: BWindow(a_frame, GetLeafName(a_filename), a_look, a_feel, a_flags, a_workspace)
{
	FDWConstructorWork(a_filename);
}


void FileDocumentWindow::FDWConstructorWork(const char* a_filename)
{
	//If be_app is a DocumentApp, register the window with be_app
	DocumentApp* document_app = cast_as(be_app,DocumentApp);
	if(document_app)
		document_app->RegisterWindow();
	m_filename = new char[strlen(a_filename)+1];
	strcpy(m_filename,a_filename);
	m_quit_after_save = false;
	m_modified = false;
	m_save_panel = NULL;
}


FileDocumentWindow::~FileDocumentWindow()
{
	//Delete the save panel
	if(m_save_panel)
		delete m_save_panel;

	//Delete the filename storage
	delete[] m_filename;

	//Tell the app to see if it should quit
	DocumentApp* document_app = cast_as(be_app,DocumentApp);
	if(document_app)
		document_app->UnregisterWindow();
}


bool FileDocumentWindow::QuitRequested()
{
	if(m_modified)
	{
		thread_id quit_thread = spawn_thread(SavePrompt,"SavePromptThread",B_NORMAL_PRIORITY,this);
		if(quit_thread != B_NO_MORE_THREADS && quit_thread != B_NO_MEMORY)
			resume_thread(quit_thread);
		return false;
	}
	else
		return true;
}


void FileDocumentWindow::MessageReceived(BMessage *a_message)
{
	switch(a_message->what)
	{
		case FDW_SAVE_AS_REQESTED:
			if(!m_save_panel)
				CreateSavePanel();
			if(m_save_panel)
				m_save_panel->Show();
			break;
		case B_SAVE_REQUESTED:
			if(InitiateSave(a_message))
				if(m_quit_after_save)
				{
					DocumentApp* document_app = cast_as(be_app,DocumentApp);
					if(document_app)
						if(document_app->IsQuitInProgress())
							be_app->PostMessage(B_QUIT_REQUESTED);
					Quit();
				}
			break;
		case B_CANCEL:
			m_quit_after_save = false;
			break;
		default:
			BWindow::MessageReceived(a_message);
			break;
	}
}


bool FileDocumentWindow::InitiateSave(BMessage* a_message)
{
	entry_ref save_location;

	if(a_message->FindRef("directory",&save_location) != B_OK)
	{
		//User hit Save (not Save as...)
		if(strncmp(m_filename,"Untitled",8) == 0)
		{
			//It's an untitled document, so I need the save panel
			if(!m_save_panel)
				CreateSavePanel();
			if(m_save_panel)
				m_save_panel->Show();
			return false;
		}
		else
			return Save();
	}
	else
	{
		//It came from the save panel
		BEntry save_location_entry(&save_location);
		BPath save_location_path;
		if(save_location_entry.GetPath(&save_location_path) != B_NO_ERROR)
			return false;
		else
		{
			char* leaf_name;
			if(a_message->FindString("name",(const char **)&leaf_name) != B_OK)
				return false;
			else
			{
				save_location_path.Append(leaf_name);
				char* temp = m_filename;
				char* old_filename = new char[strlen(temp)+1];
				strcpy(old_filename,temp);
				SetFileName(save_location_path.Path());
				bool result = Save();
				if(result == false)
					SetFileName(old_filename);
				delete[] old_filename;
				return result;
			}
		}
	}
}


void FileDocumentWindow::SetFileName(const char* a_filename)
{
	delete[] m_filename;
	m_filename = new char[strlen(a_filename)+1];
	strcpy(m_filename,a_filename);
	SetTitle(GetLeafName());
}


void FileDocumentWindow::SetModifiedFlag()
{
	m_modified = true;
}


bool FileDocumentWindow::IsModifiedFlagSet()
{
	return m_modified;
}


void FileDocumentWindow::CreateSavePanel()
{
	//Start with the file name.  If that doesn't work, NULL == home.
	entry_ref* location_ref = NULL;
	if(strncmp(m_filename,"Untitled",8) != 0)
	{
		//It has a full path, so set that location
		BEntry the_entry(m_filename);
		BEntry the_parent;
		the_entry.GetParent(&the_parent);
		if(the_parent.InitCheck() == B_NO_ERROR)
		{
			//Successfully got the parent, so get an entry_ref for it
			location_ref = new entry_ref;
			if(the_parent.GetRef(location_ref) != B_NO_ERROR)
			{
				//Whoops, failed to get an entry_ref for the_parent
				delete location_ref;
				location_ref = NULL;
			}
		}
	}

	status_t error;
	BMessenger this_window_messenger((BHandler*)NULL,this,&error);
	if(error == B_OK)
	{
		m_save_panel = GetSavePanel(&this_window_messenger,location_ref);
		if(m_save_panel)
			m_save_panel->SetSaveText(GetLeafName());
	}
	else
		fprintf(stderr,"FileDocumentWindow failed to create a valid BMessenger for itself\n");
}


bool FileDocumentWindow::Save()
{
	return true;
}


BFilePanel* FileDocumentWindow::GetSavePanel(BMessenger* a_messenger,entry_ref* a_location_ref)
{
	return new BFilePanel(B_SAVE_PANEL,a_messenger,a_location_ref,0,false);
}


void FileDocumentWindow::QuitOnceSaved()
{
	m_quit_after_save = true;
}


const char* FileDocumentWindow::GetLeafName(const char* a_filename)
{
	const char* leaf;
	if(a_filename)
		leaf = a_filename;
	else
		leaf = m_filename;
	const char* pos = leaf;

	while(*pos != 0)
	{
		if(*pos == '/' && *(pos+1) != 0)
			leaf = pos+1;
		pos++;
	}
	return leaf;
}


const char* FileDocumentWindow::GetFileName()
{
	return m_filename;
}


int32 SavePrompt(void *a_data)
{
	FileDocumentWindow* window = (FileDocumentWindow*)a_data;
	BMessenger window_messenger(NULL,window);

	const char* text0 = "Save changes to the document \"";
	const char* text1 = window->GetLeafName();
	const char* text2 = "\"?";
	char* text = new char[strlen(text0)+strlen(text1)+strlen(text2)+1];
	strcpy(text,text0);
	strcat(text,text1);
	strcat(text,text2);

	BAlert* save_alert = new BAlert(NULL, text, "Don't save","Cancel","Save");
	int32 result = save_alert->Go();
	delete[] text;

	DocumentApp* document_app = NULL;
	switch(result)
	{
		case 0:	//Don't save
			if(!window_messenger.LockTarget())
				return 0;			// window is dead so exit
			window->Quit();
			be_app->PostMessage(B_QUIT_REQUESTED);
			break;
		case 1:	//Cancel
			document_app = cast_as(be_app,DocumentApp);
			if(document_app)
				document_app->CancelQuitProcess();
			break;
		case 2:	//Save
			if(!window_messenger.LockTarget())
				return 0;			// window is dead so exit
			window->QuitOnceSaved();
			window->PostMessage(B_SAVE_REQUESTED);
			window->Unlock();
			break;
	}
	return 0;
}
